/*
 * @brief Secondary loader main application code
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#include "chip.h"
#include "sl_common.h"
//#include <string.h>
#include "chip_setup.h"


#define LED_BLUE_ON  GPIOSetDir( 0, 11, 1 );\
					GPIOSetBitValue(0, 11, 0);
					
#define LED_BLUE_OFF GPIOSetDir( 0, 11, 1 );\
					GPIOSetBitValue(0, 11, 1);
					
#define LED_RED_ON  GPIOSetDir( 0, 13, 1 );\
					GPIOSetBitValue(0, 13, 0);
					
#define LED_RED_OFF GPIOSetDir( 0, 13, 1 );\
					GPIOSetBitValue(0, 13, 1);

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

typedef void (*loopHostIfFunc)(int pNum);
loopHostIfFunc loopI2C_pFunc;

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

SL_HOSTIF_PACKET_T sendBuff, recvBuff;

IMG_HEADER_T *pImageHeader1 = (IMG_HEADER_T *) (SL_BOOTAPP_ADDR1 + SL_BOOTAPP_IMGHDR_OFS);
IMG_HEADER_T *pImageHeader2 = (IMG_HEADER_T *) (SL_BOOTAPP_ADDR2 + SL_BOOTAPP_IMGHDR_OFS);

static SL_PINSETUP_T *pinCfgData = (SL_PINSETUP_T *) SL_ADDRESS_APPPINDATA;
static uint32_t *pushAppFlag = (uint32_t *) SL_ADDRESS_APPCALLEDFL;

uint32_t *FW_VERSION1=(uint32_t *) 0x2114;
uint32_t *FW_VERSION2=(uint32_t *) 0x5114;

uint8_t App_CRC_Flag1=0;
uint8_t App_CRC_Flag2=0;


/*****************************************************************************
 * Private functions
 
 ****************************************************************************/

/* It is recommended to call this function on system boot or transfer to
   the app. */
void CleanUpSystem(void)
{
	/* Disable peripheral clocks for host IF */
	switch (slIfConfig.ifSel) {
	case SL_I2C2:
		shutdownInterfaceI2C(2);
		break;

	case SL_I2C1:
		shutdownInterfaceI2C(1);
		break;

	case SL_I2C0:
		shutdownInterfaceI2C(0);
		break;
	
	default:
		break;
	}
	
	Chip_GPIO_DeInit();
	Disable_Periph_Clock(CLK_IOCON);
	
	LPC_SYSCON->MAINCLKSEL        = MAINCLKSEL_VAL;     // Update the actual register
	LPC_SYSCON->MAINCLKUEN        = 0;                  // Toggle update register
	LPC_SYSCON->MAINCLKUEN        = 1;
	
}

void hostifAutoDetect(void)
{
	int i;

	/* All I2C interfaces are enabled for now. Initial port detection uses the I2C interrupts. */	   
	Enable_Periph_Clock(CLK_SWM);
	
	/*LPC804 only have two I2C*/
	for (i = 0; i < 2; i++) {
		setupMuxingI2C(i);
		setupInterfaceI2C(i);
		startHostIfI2C(i);
	}

	NVIC_EnableIRQ(I2C0_IRQn);
	NVIC_EnableIRQ(I2C1_IRQn);

	/* Wait forever for any I2C to be detected */
	while ((slIfConfig.ifSel == SL_AUTO) ) {
		/* I2C detection occurs in the I2C interrupt handlers */
	}

	/* Shut down I2C ports and disable IRQs */
	NVIC_DisableIRQ(I2C0_IRQn);
	NVIC_DisableIRQ(I2C1_IRQn);

	for (i = 0; i < 2; i++) {
		shutdownInterfaceI2C(i);
	}

}

Bool boot_App_ImageCheck(uint8_t app_number)
{
	IMG_HEADER_T *pImageHeader;
	Bool boot = FALSE;
	
	if(app_number==1)
		pImageHeader=pImageHeader1;
	else
		pImageHeader=pImageHeader2;
		
	/* check if there is a image header in flash or else don't boot */
	if ((pImageHeader->header_marker == IMG_HEADER_MARKER) &&
		(pinCheckValidHostif(&pImageHeader->hostifPinCfg) != FALSE)) {	/* Check if a valid ping config information is programmed in the image header */

		switch (pImageHeader->hostifPinCfg.img_type) {
		case IMG_AP_WAIT:
			/* goto secondary boot loader and wait for AP BOOT command */
			boot = FALSE;
			break;

		default:
		case IMG_NO_CRC:
		case IMG_NORMAL:
			/* check if host wants us to bypass boot due to bad image */
			/* check host requesting us to halt boot through IRQ line (pulled low after reset) */

		    GPIOSetDir(PINCFG_GET_PORT(pImageHeader->hostifPinCfg.hostIrqPortPin), PINCFG_GET_PIN(pImageHeader->hostifPinCfg.hostIrqPortPin),0 );
		
		    if(GPIOGetPinValue( PINCFG_GET_PORT(pImageHeader->hostifPinCfg.hostIrqPortPin),PINCFG_GET_PIN(pImageHeader->hostifPinCfg.hostIrqPortPin)) == false) {

				/* go to secondary boot loader */
				boot = FALSE;
				break;
			}
			/* skip CRC check if image type is IMG_NO_CRC */
			if (pImageHeader->hostifPinCfg.img_type == IMG_NO_CRC) {
				/* no more checks just boot */
				boot = TRUE;
				break;
			}

		case IMG_NO_WAIT:
			/* Perform a CRC check of the FLASH where the application is located. Jump
			   to it if it is valid, but only if this wasn't called from the
			   application. */
			if (checkAppCRC(app_number) == 0) {
				/* Valid application exists in application FLASH, so it's ok to boot the
				   application. */				
				boot = TRUE;
			}

			break;

		}
	}
	return boot;

}

void boot_ImageCheck(void)
{

	/*If IRQ pin is pulled low by host, skip app CRC check and don't boot app.*/
	App_CRC_Flag1=boot_App_ImageCheck(1);
	App_CRC_Flag2=boot_App_ImageCheck(2);
	
	if(App_CRC_Flag1==TRUE && App_CRC_Flag2==TRUE)
	{
		if(*FW_VERSION1>=*FW_VERSION2)
			doCleanBoot(SL_BOOTAPP_ADDR1);
		else
			doCleanBoot(SL_BOOTAPP_ADDR2);
	}
	else if(App_CRC_Flag1==TRUE)
		doCleanBoot(SL_BOOTAPP_ADDR1);
	else if(App_CRC_Flag2==TRUE)
		doCleanBoot(SL_BOOTAPP_ADDR2);
}

/*****************************************************************************
 * Public functions
 ****************************************************************************/

 void doCleanBoot(uint32_t appBootAddr)
{
	/* Cleanup before going to app */
	CleanUpSystem();

	bootValidApp(appBootAddr);
	/***** Control should never come here *****/

}

static uint8_t isbootFromReset(void)
{
	/* Was loader booted from app (1) or from FLASH on reset (0x44)? */
	if((uint8_t)*pushAppFlag == 0x44){
		return 1;
	}

	return 0;
}

static void I2C_Interface_Detect(void)
{
	if (isbootFromReset()) {	
		
		/* Host interface pin setup not provided by app */		
		/*Configurate IRQ pin, P0_15*/			
		IRQ_Pin_Config(&slIfConfig.hostIrqPortPin);

		/*Dectect which I2C is used*/
		//hostifAutoDetect();	
		slIfConfig.ifSel = SL_I2C0;
	
	}
	else {
		if(pinCheckValidHostif(pinCfgData))
		{
			/* App provided the host interface pin setup, so use it to set the version this app uses */		   
			slIfConfig.ifSel = pinCfgData->ifSel;

			/* App defined pins--IRQ pin*/
			parsePortData(&slIfConfig.hostIrqPortPin, pinCfgData->hostIrqPortPin);
		}
	}
	
	/*Set IRQ Pin as OUTPUT*/
	GPIOSetDir( slIfConfig.hostIrqPortPin.port, slIfConfig.hostIrqPortPin.pin, 1 );
	Hostif_AssertIRQ();
}

static uint8_t I2C_Interface_Config()
{
	uint8_t i2c_num=0;

	switch (slIfConfig.ifSel) {

	case SL_I2C1:
		i2c_num = 1;
		setupMuxingI2C(i2c_num);
		setupInterfaceI2C(i2c_num);
		startHostIfI2C(i2c_num);
		loopI2C_pFunc = &loopHostIfI2C;
		slIfConfig.pI2C = LPC_I2C1;
		break;

	case SL_I2C0:
		i2c_num = 0;
		setupMuxingI2C(i2c_num);
		setupInterfaceI2C(i2c_num);
		startHostIfI2C(i2c_num);
		loopI2C_pFunc = &loopHostIfI2C;
		slIfConfig.pI2C = LPC_I2C0;
		break;
	}
	return i2c_num;
}



/**
 * @brief	main routine for Secondary loader
 * @return	Function should not exit
 */

int main(void)
{
	IRQn_Type irqNum;	
	int pNum;
	
	Enable_Periph_Clock(CLK_SWM);
	Enable_Periph_Clock(CLK_IOCON);
	
	GPIOInit();
	LED_BLUE_OFF;
	
	if (isbootFromReset()) {	          
		/* Check image in flash and boot *///  Maybe jump to application area.
		boot_ImageCheck();
	}
	
	/* Disable sysTick */
	SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

	/* Disable all NVIC interrupts in case this loader is called from the user application */	   
	for (irqNum = SPI0_IRQn; irqNum <= PININT7_IRQn; irqNum++) {
		NVIC_DisableIRQ(irqNum);
	}

	/* Enable core interrupts */
	__enable_irq();

	/* Update core clock variable */
	SystemCoreClockUpdate();

	/* Initialize host interface */
	Hostif_Init();

	/*Detect which I2C is used*/
  I2C_Interface_Detect();

	/*Configurate I2C Interface*/
	pNum=I2C_Interface_Config();

	LED_BLUE_ON;

	/********************************************************/
	/****** Start of main host interface handling */
	/********************************************************/

	/* Wait for packets from the host, process the packet, and then respond to the host */
	while (1) {
		
		loopI2C_pFunc(pNum);

		/* Is packet pending? */
		if (hostIfPacketPending() == true) {
			/* Send packet to parser and get new packet */
			processHostIfPacket(&recvBuff, &sendBuff);

			/* Handle response */
			while (hostIfPacketPending() != true) {
				loopI2C_pFunc(pNum);
			}
		}
	}

	/********************************************************/
	/****** End of main host interface handling */
	/********************************************************/

	/* Never called */
	return 0;
}
